// migrate.js
import { resolve, dirname } from "path";
import { fileURLToPath } from "url";
import dotenv from "dotenv";
import sharp from "sharp";
import { createClient } from "@supabase/supabase-js";
import pg from "pg";

const { Pool } = pg;
const __dirname = dirname(fileURLToPath(import.meta.url));
dotenv.config({ path: resolve(__dirname, "../../../.env") });

// === ENV CHECK ===
console.log("[ENV CHECK]");
[
  "NEXT_PUBLIC_SUPABASE_URL",
  "SUPABASE_SERVICE_ROLE_KEY",
  "DATABASE_URL",
].forEach((key) => {
  console.log(`${key}: ${process.env[key] ? "✅ OK" : "❌ MISSING"}`);
});
if (
  !process.env.NEXT_PUBLIC_SUPABASE_URL ||
  !process.env.SUPABASE_SERVICE_ROLE_KEY ||
  !process.env.DATABASE_URL
) {
  throw new Error("❌ Missing environment variables. Please check .env");
}

const SUPABASE_URL = process.env.NEXT_PUBLIC_SUPABASE_URL;
const SUPABASE_SERVICE_KEY = process.env.SUPABASE_SERVICE_ROLE_KEY;
const BUCKET_NAME = "application-drive";
const supabase = createClient(SUPABASE_URL, SUPABASE_SERVICE_KEY);
const pool = new Pool({ connectionString: process.env.DATABASE_URL });

// Detect file type by base64 header
function detectFileType(base64Data) {
  const signatures = [
    { sig: "/9j/", mime: "image/jpeg", ext: "jpeg" },
    { sig: "iVBORw0KGgo", mime: "image/png", ext: "png" },
    { sig: "JVBER", mime: "application/pdf", ext: "pdf" },
    {
      sig: "UEsDB",
      mime: "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
      ext: "docx",
    },
  ];
  for (const s of signatures) {
    if (base64Data.startsWith(s.sig)) return s;
  }
  return { mime: "application/octet-stream", ext: "bin" };
}

async function compressIfImage(buffer, mime) {
  if (mime.startsWith("image/")) {
    return sharp(buffer).jpeg({ quality: 80 }).toBuffer();
  }
  return buffer;
}

async function fileExists(filePath) {
  const { data, error } = await supabase.storage
    .from(BUCKET_NAME)
    .list(filePath.substring(0, filePath.lastIndexOf("/")) || "", {
      search: filePath.split("/").pop(),
    });
  if (error) {
    console.error(
      `⚠️ Error checking file existence for ${filePath}:`,
      error.message
    );
    return false;
  }
  return data && data.length > 0;
}

async function migrate() {
  const client = await pool.connect();
  let successCount = 0;
  let failCount = 0;
  let skipCount = 0;

  try {
    // 1️⃣ Get existing columns in the table
    const { rows: columnRows } = await client.query(`
      SELECT column_name 
      FROM information_schema.columns
      WHERE table_name = 'student_applications'
    `);
    const existingColumns = new Set(columnRows.map((r) => r.column_name));

    // 2️⃣ Get all records
    const columnsToMigrate = [
      "profile_photo",
      "ResultQualification",
      "english_result_file",
      "Passport_FPage",
      "Healthdeclare_file",
      "payment_receipt_file",
      "declaration_signature",
      "graduatesecondry",
      "Cert_sec_stud",
      "Certificate_completion_studies_file",
      "second_payment",
      "chsi",
      "cscse",
      "pre_chsi",
      "pre_cscse",
      "eval_url",
      "offer_letter_url",
      "emgs_url",
    ];

    const { rows } = await client.query(`
      SELECT id, user_id, user_email, ${columnsToMigrate.join(", ")}
      FROM student_applications
    `);

    console.log(`[INFO] Found ${rows.length} records.`);

    for (const row of rows) {
      const userFolder = row.user_id || row.user_email || "unknown_user";

      for (const col of columnsToMigrate) {
        if (!existingColumns.has(col)) {
          console.log(`⚠️ Skipping missing column: ${col}`);
          skipCount++;
          continue;
        }

        const base64Data = row[col];
        if (!base64Data) {
          skipCount++;
          continue;
        }

        try {
          const cleaned = base64Data.replace(/^data:.*;base64,/, "");
          const fileType = detectFileType(cleaned);
          const buffer = Buffer.from(cleaned, "base64");
          const finalBuffer = await compressIfImage(buffer, fileType.mime);

          const filePath = `${userFolder}/${row.id}/${col}.${fileType.ext}`;

          // 🛑 Check if file already exists
          if (await fileExists(filePath)) {
            console.log(`⏩ Skipping upload, file already exists: ${filePath}`);
            skipCount++;
            continue;
          }

          // Upload if not exists
          const { error: uploadError } = await supabase.storage
            .from(BUCKET_NAME)
            .upload(filePath, finalBuffer, {
              contentType: fileType.mime,
              upsert: false, // prevent overwrite
            });

          if (uploadError) {
            console.error(
              `❌ Upload failed for ${filePath}:`,
              uploadError.message
            );
            failCount++;
            continue;
          }

          const { data: publicUrlData } = supabase.storage
            .from(BUCKET_NAME)
            .getPublicUrl(filePath);

          await client.query(
            `UPDATE student_applications SET ${col} = $1 WHERE id = $2`,
            [publicUrlData.publicUrl, row.id]
          );

          console.log(`✅ ${col} for record ${row.id} migrated to ${filePath}`);
          successCount++;
        } catch (err) {
          console.error(
            `Error processing ${col} for record ${row.id}:`,
            err.message
          );
          failCount++;
        }
      }
    }

    // 3️⃣ Summary log
    console.log("\n=== Migration Summary ===");
    console.log(`✅ Success: ${successCount}`);
    console.log(`❌ Fail: ${failCount}`);
    console.log(`⚠️ Skipped: ${skipCount}`);
    console.log("=========================");
  } finally {
    client.release();
  }
}

migrate().then(() => {
  console.log("Migration complete.");
  process.exit(0);
});
