const loginDetail = JSON.parse(sessionStorage.getItem("loginDetail"));
const username = loginDetail ? loginDetail.username : "";
const role = loginDetail ? loginDetail.role : "";

const supabase = window.supabase;

async function fetchStudentNames() {
  const storedLoginDetail = JSON.parse(sessionStorage.getItem("loginDetail"));
  let agent_name = "";

  if (storedLoginDetail) {
    agent_name = storedLoginDetail.full_name;
    document.getElementById("userFullName").innerText = agent_name;
  } else {
    console.error("Login detail missing");
    return;
  }

  const { data, error } = await supabase
    .from("student_applications")
    .select(
      "id, full_name, status, remark, second_payment, offer_letter_url, emgs_url, eval_url"
    )
    .eq("register_by", agent_name);

  if (error) {
    console.error("Error fetching student names:", error.message);
    return;
  }

  const tbody = document.querySelector("tbody");
  tbody.innerHTML = "";
  const fragment = document.createDocumentFragment();

  for (const [index, student] of data.entries()) {
    const status = student.status ?? "new";
    const remark = student.remark ?? "";
    const badgeClass = getStatusBadgeClass(status);

    const row = document.createElement("tr");
    row.innerHTML = `
      <td class="text-sm text-center">${index + 1}</td>
      <td class="text-sm text-left" style="padding-left: 20px;"><div class="px-2 py-1"><div class="my-auto"><h6 class="mb-0 text-sm" style="text-align: left;">${
        student.full_name
      }</h6></div></div></td>
      <td style="text-align: left;"> <span class="badge ${badgeClass} text-xs">${status.replace(
      "_",
      " "
    )}</span></td>
    `;

    const receiptTd = document.createElement("td");
    receiptTd.className = "text-sm text-center";

    if (status && status.toLowerCase() === "pre_approved") {
      const group = document.createElement("div");
      group.className = "file-group d-flex align-items-center gap-2 w-100";

      group.innerHTML = `
    <input type="file" accept="image/*,application/pdf" class="form-control form-control-sm receipt-input" data-id="${student.id}" />
    <span class="material-symbols-rounded text-success upload-receipt-icon" style="font-size: 24px; cursor: pointer;">upload</span>
    <span class="receipt-preview-container" style="margin-left: 10px;"></span>
  `;

      receiptTd.appendChild(group);

      const previewContainer = group.querySelector(
        ".receipt-preview-container"
      );

      // Show existing uploaded receipt
      if (student.second_payment) {
        if (student.second_payment.endsWith(".pdf")) {
          previewContainer.innerHTML = `
        <a href="${student.second_payment}" target="_blank" class="badge bg-primary text-white px-2 py-1 rounded">View PDF</a>
      `;
        } else {
          previewContainer.innerHTML = `
        <img src="${student.second_payment}" alt="Receipt preview" style="max-height: 50px; border: 1px solid #ccc; border-radius: 4px;" />
      `;
        }
      }

      // Preview for newly selected file
      const fileInput = group.querySelector(".receipt-input");
      fileInput.addEventListener("change", function () {
        const file = fileInput.files[0];
        if (!file) return;

        previewContainer.innerHTML = "";

        if (file.type === "application/pdf") {
          previewContainer.innerHTML = `<span class="badge bg-info text-dark px-2 py-1 rounded">PDF Selected</span>`;
        } else if (file.type.startsWith("image/")) {
          const reader = new FileReader();
          reader.onload = function (e) {
            previewContainer.innerHTML = `<img src="${e.target.result}" alt="Preview" style="max-height: 50px; border: 1px solid #ccc; border-radius: 4px;" />`;
          };
          reader.readAsDataURL(file);
        }
      });
    } else {
      receiptTd.innerHTML = `<span class="text-muted">-</span>`;
    }
    row.appendChild(receiptTd);

    const remarkTd = document.createElement("td");
    remarkTd.className = "text-sm text-left";
    remarkTd.style.whiteSpace = "normal";
    remarkTd.style.wordBreak = "break-word";
    remarkTd.style.overflowWrap = "break-word";
    remarkTd.style.maxWidth = "300px"; // Optional: control width
    remarkTd.style.textAlign = "justify"; // Justify the text
    remarkTd.innerText = remark;
    row.appendChild(remarkTd);

    const offerLetterTd = document.createElement("td");
    offerLetterTd.className = "text-sm text-center";

    const hasOfferZip =
      status &&
      ["pre_approved", "approved"].includes(status.toLowerCase()) &&
      student.offer_letter_url;

    if (hasOfferZip) {
      offerLetterTd.innerHTML = `
        <span class="material-symbols-rounded download-offer" style="font-size:24px; cursor:pointer;"
          data-id="${student.id}"
          data-offer="${student.offer_letter_url}"
          title="Download Offer Letter ZIP">
          download
        </span>
      `;
    } else {
      offerLetterTd.innerHTML = `<span class="text-muted">-</span>`;
    }
    row.appendChild(offerLetterTd);

    const filesTd = document.createElement("td");
    filesTd.className = "text-sm text-center";
    const hasEMGSAndEval =
      status &&
      status.toLowerCase() === "approved" &&
      student.emgs_url &&
      student.eval_url;

    if (hasEMGSAndEval) {
      filesTd.innerHTML = `
        <span class="material-symbols-rounded download-icon" style="font-size:24px; cursor:pointer;" 
          data-id="${student.id}"
          data-emgs="${student.emgs_url}" 
          data-eval="${student.eval_url}" 
          title="Download EMGS and EVAL">
          folder_zip
        </span>
      `;
    } else {
      filesTd.innerHTML = `<span class="text-muted">-</span>`;
    }
    row.appendChild(filesTd);

    fragment.appendChild(row);
  }

  tbody.appendChild(fragment);
}

document.addEventListener("click", async (e) => {
  const target = e.target;

  if (target.classList.contains("upload-receipt-icon")) {
    const parent = target.closest(".file-group");
    const input = parent.querySelector(".receipt-input");
    const previewContainer = parent.querySelector(".receipt-preview-container");

    const studentId = input.dataset.id;
    const file = input.files[0];

    if (!file) {
      alert("❗ Please select a file first.");
      return;
    }

    const reader = new FileReader();

    reader.onload = async () => {
      const base64String = reader.result;

      const { error } = await supabase
        .from("student_applications")
        .update({
          second_payment: base64String,
          status: "pre_processing", // status transition
        })
        .eq("id", studentId);

      if (error) {
        console.error("Upload failed:", error.message);
        alert("❌ Upload failed.");
        return;
      }

      // Update preview
      if (file.type === "application/pdf") {
        previewContainer.innerHTML = `<a href="${base64String}" target="_blank" class="badge bg-primary text-white px-2 py-1 rounded">View PDF</a>`;
      } else {
        previewContainer.innerHTML = `<img src="${base64String}" alt="Receipt preview" style="max-height: 50px; border: 1px solid #ccc; border-radius: 4px;" />`;
      }

      alert("✅ Receipt uploaded. Status updated to 'pre_processing'.");
      fetchStudentNames(); // Refresh UI
    };

    reader.readAsDataURL(file);
  }

  if (target.classList.contains("download-icon")) {
    const id = target.dataset.id;
    const photo = await getImageDetail(id);
    if (photo) generatePDFimg(photo);
    return;
  }

  if (target.classList.contains("download-offer")) {
    const base64 = target.dataset.offer;
    const studentId = target.dataset.id;

    const byteCharacters = atob(base64);
    const byteNumbers = new Array(byteCharacters.length);
    for (let i = 0; i < byteCharacters.length; i++) {
      byteNumbers[i] = byteCharacters.charCodeAt(i);
    }
    const byteArray = new Uint8Array(byteNumbers);
    const blob = new Blob([byteArray], { type: "application/zip" });

    const link = document.createElement("a");
    link.href = URL.createObjectURL(blob);
    link.download = `offer_letter_${studentId}.zip`;
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  }
});

async function getImageDetail(studentId) {
  const { data, error } = await window.supabase
    .from("student_applications")
    .select("emgs_url, eval_url")
    .eq("id", studentId)
    .single();

  if (error) {
    console.error("Error fetching student letter:", error.message);
    alert("❌ Error fetching student letter.");
    return null;
  }
  return data;
}

function generatePDFimg(photo) {
  const { jsPDF } = window.jspdf;
  const doc = new jsPDF();

  const imageFields = [
    { key: "emgs_url", label: "" },
    { key: "eval_url", label: "" },
  ];

  const pageWidth = doc.internal.pageSize.getWidth();
  const pageHeight = doc.internal.pageSize.getHeight();

  let firstPage = true;

  imageFields.forEach(({ key, label }) => {
    if (photo[key]) {
      if (!firstPage) {
        doc.addPage();
      } else {
        firstPage = false;
      }

      doc.setFontSize(12);
      doc.text(label, 10, 10);
      doc.addImage(photo[key], "JPEG", 0, 20, pageWidth, pageHeight - 20);
    }
  });

  doc.save("Letter_Application.pdf");
}

function getStatusBadgeClass(status) {
  switch (status.toLowerCase()) {
    case "approved":
      return "bg-success text-white";
    case "pre_approved":
      return "bg-primary text-white";
    case "pre_processing":
      return "bg-info text-white";
    case "rejected":
      return "bg-danger text-white";
    case "new":
    default:
      return "bg-warning text-dark";
  }
}

document.addEventListener("DOMContentLoaded", fetchStudentNames);
